<?php
/******************************************************************************
 *         FILE: oci8_class.php
 *      VERSION: 0.46 (09-AUG-2004)
 *       AUTHOR: Sascha 'SieGeL' Pfalz
 *      PURPOSE: Database class using Oracle 8i+ API (OCI)
 * REQUIREMENTS: dbdefs.inc.php for global access data (user,pw,host,appname)
 *        NOTES: Uses only one DB link ($this->sock) but returns sock on Connect()!
 *      METHODS: Connect() / Disconnect()
 *               Print_Error()
 *               Query() / QueryResult() / FetchResult() / FreeResult()
 *               Version() / GetClassVersion() / GetQueryCount()
 *               Commit() / Rollback()
 *               SetDebug() / PrintDebug() / SQLDebug()
 *               GetBindVars() - PRIVATE!
 *               DescTable()
 *               Prepare() / Execute()
 *               SearchQueryCache() / RemoveFromQueryCache()
 *               checkSock()
 *               GetConnectionHandle()
 *               QueryInOutHash()
 *******************************************************************************/

// Debug related defines:

define(DBOF_DEBUGOFF    , (1 << 0));
define(DBOF_DEBUGSCREEN , (1 << 1));
define(DBOF_DEBUGFILE   , (1 << 2));

// These defines are used in DescTable():

define(DBOF_COLNAME, 0);
define(DBOF_COLTYPE, 1);
define(DBOF_COLSIZE, 2);
define(DBOF_COLPREC, 3);

// Used for Query Cache (V0.38)+:

define(DBOF_CACHE_QUERY     , 0);
define(DBOF_CACHE_STATEMENT , 1);

// Class code starts here

class db_oci8
  {
  var $sock;
  var $host;
  var $user;
  var $password;
  var $database;
  var $querycounter;
  var $stmt;
  var $appname;
  var $debug;
  var $sqlerr;
  var $sqlcache;
  var $errvars;
  var $no_exit;
  var $sqlcount;
    
  /************************************************************************
   * Constructor 
   ************************************************************************/
  
  function db_oci8()
    {
    require_once('dbdefs.inc.php');
    $this->classversion = "0.46";   // Version of our class
    $this->host         = "";       // TNS Name of DB to connect to
    $this->user         = "";       // Username
    $this->pass         = "";       // Password 
    $this->appname      = APPNAME;  // Name of our Application
    $this->database     = "";       // Oracle does not use this
    $this->sock         = 0;        // Internal database handle 
    $this->querycounter = 0;        // How many queries where executed
    $this->stmt         = 0;        // Oracle Statement handler
    $this->debug        = 0;        // Debug is off per default
    $this->sqlcache     = array();  // Internal SQL cache for Prepare()/Execute()
    $this->sqlerr       = "";       // Contains possible SQL query that failed
    $this->errvars      = array();  // All passed variables except QUERY and Flags
    $this->no_exit      = 0;        // Flag for Prepare/Execute pair to indicate if we should exit
    $this->sqlcount     = 0;        // Counter for Prepare/Execute pair to reference correct query
    }

  /************************************************************************
   * Performs the connection to Oracle. 
   * If anything goes wrong calls Print_Error()
   * Calls also a stored procedure to register Appname. This helps DBAs 
   * to better fine tune their databases according to application needs.
   ************************************************************************/
  
  function Connect($user="",$pass="",$host="",$db="")
    {
    if($this->sock) return($this->sock);
    if($user!="") $this->user = $user;
    else $this->user = DB_USER;
    if($pass!="") $this->pass = $pass;
    else $this->pass = DB_PASS;
    if($host!="") $this->host = $host;
    else $this->host = DB_HOST;
    $this->sock = @OCILogon($this->user,$this->pass,$this->host);
    if(!$this->sock)
      {
      $this->Print_Error('Connection to "'.$this->host.'" failed!');
      return(0);
      }
    $connquery = "BEGIN ";
    if(!defined('DB_NUM_DECIMAL') || !defined('DB_NUM_GROUPING')) 
      {
      $this->Print_Error("You have to define DB_NUM_DECIMAL/DB_NUM_GROUPING in dbdefs.inc.php first !!!");
      }
    if(DB_REGISTER) 
      { 
      $connquery.= " DBMS_APPLICATION_INFO.SET_MODULE('".$this->appname."',NULL);";
      }
    $connquery.= " EXECUTE IMMEDIATE 'ALTER SESSION SET NLS_NUMERIC_CHARACTERS = ''".DB_NUM_DECIMAL.DB_NUM_GROUPING."''';";
    $connquery.= " END;";
    $this->Query($connquery,OCI_ASSOC,0);
    return($this->sock);        
    }

  /************************************************************************
   * Disconnects from Oracle. 
   * You may optionally pass an external link identifier
   ************************************************************************/

  function Disconnect($other_sock=-1)
    {
    if($other_sock!=-1) @OCILogoff($other_sock);
    else 
      {
      if($this->sock) 
        {
        @OCILogoff($this->sock); 
        $this->sock = 0;
        }
      }
    }

  /************************************************************************
   * Prints out Oracle Error in own <div> container and exits.
   ************************************************************************/

  function Print_Error($ustr="",$var2dump="")
    {
    if($this->stmt) 
      {
      $earr = @OCIError($this->stmt);
      } 
    elseif($this->sock)
      {
      $earr = @OCIError($this->sock);
      } 
    else
      {
      $earr = @OCIError();
      }
    $errstr   = $earr['message'];
    $errnum   = $earr['code'];
    $sqltext  = $earr['sqltext'];
    $sqlerrpos= (int)$earr['offset'];
    if($sqltext=="") 
      {
      if($this->sqlerr!='') $sqltext = $this->sqlerr;
      else $sqltext = 'N/A';
      }
    $filename = basename($_SERVER['SCRIPT_FILENAME']);
    $this->Disconnect();
    echo("<br>\n<div align=\"left\" style=\"background-color: #EEEEEE; color:#000000\" class=\"TB\">\n");
    echo("<font color=\"red\" face=\"Arial, Sans-Serif\" size=\"4\"><b>Database Error occured!</b></font><br>\n<code>\n");
    if($errnum == '') $errnum = 'N/A';
    if($errstr == '') $errstr = 'N/A';
    echo("CODE: ".$errnum."<br>");
    echo("DESC: ".$errstr."<br>");
    echo("FILE: ".$filename."<br>");
    if($ustr!='')
      {
      echo("INFO: ".$ustr."<br>");
      }
    if($sqlerrpos)
      {
      $dummy = substr($sqltext,0,$sqlerrpos);
      $dummy.='<font color="red">'.substr($sqltext,$sqlerrpos).'</font>';
      $errquery = $dummy;  
      }
    else
      {
      $errquery = $sqltext;
      } 
    echo("&nbsp;SQL: ".$errquery."<br>");
    echo("&nbsp;POS: ".$sqlerrpos."<br>");
    echo("QCNT: ".$this->GetQueryCount()."<br>");
    if(count($this->errvars))
      {
      echo("VALS: ");
      for($i = 0; $i < count($this->errvars); $i++)
        {
        $errbuf.=sprintf("P%d='%s',",($i+1),$this->errvars[$i]); 
        }  
      $errbuf = substr($errbuf,0,strlen($errbuf)-1);
      echo($errbuf."<br>");
      }
    if($var2dump!='')
      {
      echo("Dump: <pre>");
      print_r($var2dump);
      echo("</pre>");
      }
    echo("</code>\n");
    echo("</div>\n");
    @error_log($this->appname.': Error in '.$filename.': '.$ustr.' ('.chop($errstr).')',0);   
    exit;
    }

  /************************************************************************
   * Performs a single row query. Resflag can be OCI_NUM or OCI_ASSOC
   * depending on what kind of array you want to be returned.
   * $querystring => The query to be executed against the RDBMS
   * $resflag     => OCI_NUM for numeric array or OCI_ASSOC (default)
   *                 for associative array result
   * $no_exit     => 1 => Function returns errorcode instead of calling Print_Error()
   *                 0 => Will always call Print_Error()
   * Remember to pass all required variables for all defined bind vars after
   * the query, else you will recieve errors because of wrong parameter count!
   ************************************************************************/

  function Query($querystring, $resflag = OCI_ASSOC, $no_exit = 0)
    {
    $resarr         = array();
    $this->errvars  = array();
    $funcargs       = @func_num_args();
    $this->sqlerr   = $querystring;

    $this->checkSock();
    if($funcargs > 3)
      {
      $this->errvars = array_slice(func_get_args(),3);
      $res = $this->GetBindVars($querystring);
      if(($funcargs-3) != count($res))
        {
        $this->Print_Error("Query(): Parameter count does not match bind var count in query! (Defined: ".count($res)." - Supplied: ".($funcargs).")");
        }
      }      
    if($this->debug)
      {
      $this->PrintDebug($querystring);  
      }
    $this->stmt = @OCIParse($this->sock,$querystring);
    if(!$this->stmt)
      {
      $this->Print_Error('Query(): Parse failed!');
      }
    if($funcargs > 3)
      {
      for($i = 3; $i < $funcargs; $i++)
        {
        $arg[$i] = @func_get_arg($i);
        @OCIBindByName($this->stmt,$res[$i-3],$arg[$i],-1);
        }
      }
    if(!@OCIExecute($this->stmt,OCI_DEFAULT))
      {
      if($no_exit)
        {
        $err = @OCIError($this->stmt);
        return($err['code']);
        }
      else
        {
        $this->Print_Error('Query(): Execute failed!');
        }
      }
    $this->querycounter++;
    if(StriStr(substr($querystring,0,6),"SELECT"))
      {
      @OCIFetchInto($this->stmt,$resarr,$resflag+OCI_RETURN_NULLS+OCI_RETURN_LOBS);
      }
    else
      {
      $res = 0;         
      }
    @OCIFreeStatement($this->stmt);
    $this->stmt = 0;
    $this->errvars = array();
    return($resarr);
    }

  /************************************************************************
   * Performs a query and returns result. Required if you want to fetch many
   * data rows. Does not return in case of error, so no further checking
   * is required. Supports also binding, see Query() for further details
   ************************************************************************/

  function QueryResult($querystring)
    {
    $funcargs       = @func_num_args();
    $this->sqlerr   = $querystring;
    $this->errvars  = array();

    $this->checkSock();
    if($funcargs > 1)
      {
      $this->errvars = array_slice(func_get_args(),1);
      $res = $this->GetBindVars($querystring);
      if(($funcargs-1) != count($res))
        {
        $this->Print_Error("QueryResult(): Parameter count does not match bind var count in query! (Defined:".count($res)." - Supplied: ".($funcargs).")");
        }
      }      
    if($this->debug)
      {
      $this->PrintDebug($querystring);  
      }
    $this->stmt = @OCIParse($this->sock,$querystring);
    if(!$this->stmt)
      {
      $this->Print_Error('QueryResult(): Parse failed!');
      }
    @OCISetPrefetch($this->stmt,20);
    
    // If we have any of the bind vars given, bind them NOW:
    
    if($funcargs > 1)
      {
      for($i = 1; $i < $funcargs; $i++)
        {
        $arg[$i] = @func_get_arg($i);
        @OCIBindByName($this->stmt,$res[$i-1],$arg[$i],-1);
        }
      }
    if(!@OCIExecute($this->stmt,OCI_DEFAULT))
      {
      $this->Print_Error('QueryResult(): Execute failed!');
      }  
    $this->querycounter++;
    $this->sqlcache[$this->sqlcount][DBOF_CACHE_QUERY]     = $querystring;
    $this->sqlcache[$this->sqlcount][DBOF_CACHE_STATEMENT] = $this->stmt;
    $this->sqlcount++;
    return($this->stmt);
    }

  /************************************************************************
   * Returns either numeric (OCI_NUM) or associative (OCI_ASSOC) array
   * for one data row as pointed to by internal result var.
   ************************************************************************/

  function FetchResult($resflag = OCI_ASSOC,$extstmt = -1)
    {
    if($extstmt == -1)
      {
      $mystate = $this->stmt;
      }
    else
      {
      $mystate = $extstmt;
      }
    @OCIFetchInto($mystate, $res, $resflag+OCI_RETURN_NULLS+OCI_RETURN_LOBS);
    return($res);
    }

  /************************************************************************
   * Frees result obtained by QueryResult():
   ************************************************************************/

  function FreeResult($extstmt = -1)
    {
    if($extstmt == -1)
      {
      $mystate = $this->stmt;
      }
    else
      {
      $mystate = $extstmt;
      $fq = $this->SearchQueryCache($extstmt);
      if($fq != -1)
        {
        $this->RemoveFromQueryCache($fq);
        }
      }
    $this->errvars = array();
    $this->no_exit = 0;
    return(@OCIFreeStatement($mystate));
    }
  
  /************************************************************************
   * Returns Oracle Server Version. 
   * Opens an own connection if no active one exists.
   ************************************************************************/

  function Version()
    {
    $weopen = 0;
    if(!$this->sock) 
      {
      $this->Connect();
      $weopen = 1;
      }
    if($this->debug)
      {
      $this->PrintDebug('Version() called - Self-Connect: '.$weopen);
      }
    $ver = @OCIServerVersion($this->sock);
    $ret = explode("-",$ver);
    if($weopen) $this->Disconnect();
    $this->querycounter++;
    return(trim($ret[0]));
    }

  /************************************************************************
   * Returns amount of queries executed by this class
   ************************************************************************/

  function GetQueryCount()
    {
    if($this->debug)
      {
      $this->PrintDebug('GetQueryCount() called');
      }
    return($this->querycounter);
    }

  /************************************************************************
   * Commits current transaction
   ************************************************************************/

  function Commit()
    {
    $this->checkSock();
    if($this->debug)
      {
      $this->PrintDebug('COMMIT called');
      }
    return(@OCICommit($this->sock));
    }
  
  /************************************************************************
   * Rollback current transaction
   ************************************************************************/

  function Rollback()
    {
    $this->checkSock();
    if($this->debug)
      {
      $this->PrintDebug('Rollback called');
      }
    return(@OCIRollback($this->sock));
    }

  /************************************************************************
   * Function extracts all bind vars out of given query. To avoid wrong
   * determined bind vars this function first kills out all TO_*() functions
   * together with their (possible) format strings which results in a query
   * containing only valid bind vars, format tags or other similar constructed 
   * tags are removed.
   * Returns an array with all found bind vars in the order they are defined
   * inside the query.  
   ************************************************************************/

  function GetBindVars($query)
    {
    $pattern = array("/(TO_.*?\()(.*?)(,)(.*?\))/is","/(TO_.*\('.*'\))/is","/(TO_.*\()(.*\))/is");
    $replace = array("$2","","$2");
    $mydummy = $query;    // Make copy of current SQL

    $mydummy = @preg_replace($pattern,$replace,$mydummy);
    @preg_match_all("/[,|\W]?(:\w+)[,|\W]?/i",$mydummy,$res);
    return($res[1]);    
    }

  /************************************************************************
   * Function allows debugging of SQL Queries. $state can have these values:
   * DBOF_DEBUGOFF    = Turn off debugging
   * DBOF_DEBUGSCREEN = Turn on debugging on screen (every Query will be dumped on screen)
   * DBOF_DEBUFILE    = Turn on debugging on PHP errorlog
   * You can mix the debug levels by adding the according defines!
   ************************************************************************/

  function SetDebug($state)
    {
    $this->debug = $state;  
    }

  /************************************************************************
   * Handles output according to internal debug flag.
   ************************************************************************/

  function PrintDebug($msg)
    {
    if(!$this->debug) return;
    $errbuf = "";
    if($this->errvars)
      {
      $errbuf = "<br>VARS: ";
      for($i = 0; $i < count($this->errvars); $i++)
        {
        $errbuf.=sprintf("P%d='%s',",($i+1),$this->errvars[$i]); 
        }  
      $errbuf = substr($errbuf,0,strlen($errbuf)-1);
      $errbuf.="<br>";
      }
    if($this->debug & DBOF_DEBUGSCREEN) 
      {
      printf("<div align=\"left\" style=\"background-color:#ffffff; color:#000000\"><pre>DEBUG: %s%s</pre></div>\n",$msg,$errbuf);
      }      
    if($this->debug & DBOF_DEBUGFILE)
      {
      error_log('DEBUG: '.$msg,0);
      }
    }

  /************************************************************************
   * Allows to en- or disable the SQL_TRACE feature of Oracle. Pass TRUE
   * to enable or FALSE to disable. When enabled all Statements of your 
   * session are saved in a tracefile stored in 
   * $ORACLE_BASE/admin/<DBNAME>/udump/*.trc
   * After your session disconnects use the tkprof tool to generate
   * Human-readable output from the tracefile, i.e.:
   * $> tkprof oracle_ora_7527.trc out.txt
   * Now read 'out.txt' and see what happen in Oracle!
   ************************************************************************/
 
  function SQLDebug($state)
    {
    switch($state)
      {
      case  TRUE:   $sdebug = 'TRUE';
                    break;
      case  FALSE:  $sdebug = 'FALSE';
                    break;
      default:      return;
      }
    if($this->sock)
      {
      $this->Query('ALTER SESSION SET SQL_TRACE = '.$sdebug);  
      }
    }

  /************************************************************************
   * Returns version of this class
   ************************************************************************/
  
  function GetClassVersion()
    {
    return($this->classversion);
    }   

  /************************************************************************
   * Describes a table by returning an array with all table info
   ************************************************************************/

  function DescTable($tablename)
    {
    $retarr = array();
    $weopen = 0;
    if(!$this->sock) 
      {
      $this->Connect();
      $weopen = 1;
      }
    if($this->debug)
      {
      $this->PrintDebug('DescTable('.$tablename.') called - Self-Connect: '.$weopen);
      }
    $stmt = @OCIParse($this->sock,"SELECT * FROM ".$tablename." WHERE ROWNUM < 1");
    @OCIExecute($stmt);    
    $ncols = @OCINumCols($stmt);
    for ($i = 1; $i <= $ncols; $i++) 
      {
      $retarr[$i-1][DBOF_COLNAME] = @OCIColumnName($stmt, $i);
      $retarr[$i-1][DBOF_COLTYPE] = @OCIColumnType($stmt, $i);
      $retarr[$i-1][DBOF_COLSIZE] = @OCIColumnSize($stmt, $i);
      $retarr[$i-1][DBOF_COLPREC] = @OCIColumnPrecision($stmt,$i);
      }
    @OCIFreeStatement($stmt);
    if($weopen) $this->Disconnect();
    return($retarr);
    }

  /************************************************************************
   * Preparses a query but do not execute it (yet). This allows to
   * use a compiled query inside loops without having to parse it everytime
   * V0.38: All prepared() queries will be put into our own QueryCache() so
   * we can use the Prepare()/Execute() pair for more than one query at once.
   ************************************************************************/

  function Prepare($querystring, $no_exit = 0)
    {
    $this->no_exit  = $no_exit;
    $this->sqlerr   = $querystring;

    $this->checkSock();
    $stmt = @OCIParse($this->sock,$querystring);
    if(!$stmt)
      {
      if($no_exit)
        {
        $err = @OCIError($this->sock);
        return($err['code']);
        }
      else
        {
        $this->Print_Error('Prepare(): Parse failed!');
        }
      }
    if($this->debug)
      {
      $this->PrintDebug("PREPARE: #".$this->sqlcount." ".$this->sqlerr);  
      }
    $this->sqlcache[$this->sqlcount][DBOF_CACHE_QUERY]     = $querystring;
    $this->sqlcache[$this->sqlcount][DBOF_CACHE_STATEMENT] = $stmt;
    $this->sqlcount++;
    return($stmt);
    }

  /************************************************************************
   * Executes a prepare()d statement and returns the result. You may then
   * Fetch rows with FetchResult() or call FreeResult() to free your 
   * allocated result.
   * V0.38: Execute() searches first our QueryCache before executing, this
   * way we can use almost unlimited Queries at once in the Prepare/Execute pair
   ************************************************************************/

  function Execute($stmt)
    {
    $f = $this->SearchQueryCache($stmt);
    if($f == -1)
      {
      $this->Print_Error("Cannot find query for given statement #".$stmt." inside query cache!!!");  
      }
    $this->sqlerr  = $this->sqlcache[$f][DBOF_CACHE_QUERY];
    $this->errvars = array();
    $funcargs = @func_num_args();
    if($funcargs > 1)
      {
      $this->errvars = @array_slice(@func_get_args(),1);
      $res = $this->GetBindVars($this->sqlcache[$f][DBOF_CACHE_QUERY]);
      if(($funcargs-1) != count($res))
        {
        $this->stmt = $stmt;
        $this->Print_Error("Execute(): Parameter count does not match bind var count in query! (Defined:".count($res)." - Supplied: ".($funcargs).")");
        }
      }      
    if($funcargs > 1)
      {
      for($i = 1; $i < $funcargs; $i++)
        {
        $arg[$i] = @func_get_arg($i);
        @OCIBindByName($stmt,$res[$i-1],$arg[$i],-1);
        }
      }
    if($this->debug)
      {
      $this->PrintDebug($this->sqlerr);  
      }
    if(!@OCIExecute($stmt,OCI_DEFAULT))
      {
      if($this->no_exit)
        {
        $err = @OCIError($stmt);
        return($err['code']);
        }
      else
        {
        $this->stmt = $stmt;
        $this->Print_Error('Execute(): Execute failed!');
        }
      }  
    $this->querycounter++;
    return($stmt);
    } 

  /************************************************************************
   * V0.38: Searches internal query cache for given statement id. 
   * Returns index of found statement id or -1 to indicate an error 
   ************************************************************************/
  
  function SearchQueryCache($stmt)
    {
    $f = 0;
    for($i = 0; $i < $this->sqlcount; $i++)
      {
      if($this->sqlcache[$i][DBOF_CACHE_STATEMENT] == $stmt)
        {
        return($i);
        }  
      }
    return(-1);
    }

  /************************************************************************
   * v0.38: Removes query from cache that was found by a previous call 
   * to SearchQueryCache()
   ************************************************************************/

  function RemoveFromQueryCache($nr)
    {
    $newdata = array();
    $lv = 0;
    for($i = 0; $i < $this->sqlcount; $i++)
      {
      if($i != $nr)
        {
        $newdata[$lv][DBOF_CACHE_QUERY]    = $this->sqlcache[$i][DBOF_CACHE_QUERY];
        $newdata[$lv][DBOF_CACHE_STATEMENT]= $this->sqlcache[$i][DBOF_CACHE_STATEMENT];
        $lv++;        
        }
      }
    $this->sqlcache = $newdata;
    $this->sqlcount--;
    }      

  /************************************************************************
   * v0.40: Checks if we are already connected to our database. If not stops with error
   ************************************************************************/

  function checkSock()
    {
    if(!$this->sock)
      {
      $this->Print_Error("<b>!!! NOT CONNECTED TO AN ORACLE DATABASE !!!</b>");
      }
    }      

  /************************************************************************
   * V0.41: Allows to save a file to a binary object field (BLOB)
   * $file_to_save  => Full path and filename to file to save
   * $blob_table    => Name of Table where the blobfield resides
   * $blob_field    => Name of BLOB field
   * $where_clause  => Criteria to get the right row (i.e. WHERE ROWID=ABCDEF12345)
   ************************************************************************/

  function SaveBLOB($file_to_save, $blob_table, $blob_field, $where_clause)
    {
    $this->checkSock();
    if($where_clause == '')
      {
      $this->Print_Error("SaveBLOB(): WHERE clause must be non-empty, else ALL rows would be updated!!!");
      }    
    $q1 = "UPDATE ".$blob_table." SET ".$blob_field."=EMPTY_BLOB() ".$where_clause." RETURNING ".$blob_field." INTO :oralob";
    $this->sqlerr = $q1;
    $lobptr = @OCINewDescriptor($this->sock, OCI_D_LOB);
    if(!($lobstmt = @OCIParse($this->sock,$q1)))
      {
      $this->Print_Error("SaveBLOB(): Unable to parse query !!!");
      }    
    @OCIBindByName($lobstmt, ":oralob", $lobptr, -1, OCI_B_BLOB);	
    if(!@OCIExecute($lobstmt, OCI_DEFAULT))
      {
      @OCIFreeStatement($lobstmt);
      @OCIFreeDesc($lobptr);
      $this->Print_Error("SaveBLOB(): Unable to retrieve empty LOB locator !!!");
      }
    if(!$lobptr->savefile($file_to_save))
      {
      $this->RollBack();
      @OCIFreeStatement($lobstmt);
      @OCIFreeDesc($lobptr);
      $this->Print_Error("SaveBLOB(): Cannot save LOB data !!!");
      }          
    @OCIFreeDesc($lobptr);
    @OCIFreeStatement($lobstmt);
    $this->query_counter++;
    }  

  /************************************************************************
   * V0.42: Returns current connection handle or -1 if no active handle exists
   *        Useful if you want to work with OCI* functions in parallel to this class
   ************************************************************************/

  function GetConnectionHandle()
    {
    return($this->sock);  
    }

  /************************************************************************
   * V0.44: Executes a query with parameters passed as hash values. Also
   *        IN/OUT and RETURNING INTO <...> clauses are supported. You
   *        have to use FetchResult()/FreeResult() after using this function!
   ************************************************************************/

  function QueryInOutHash($query,&$inhash)
    {
    $this->checkSock();
    $this->sqlerr = $query;
    if(!($this->stmt = @OCIParse($this->sock,$query)))
      {
      $this->Print_Error("QueryInOutHash(): Unable to parse query !!!");
      }
    if(is_array($inhash))
      {
      reset($inhash);
      while(list($key,$val) = each($inhash))
        {
        @OCIBindByName($this->stmt,$key,$inhash[$key],-1);  
        }  
      }    
    if(!@OCIExecute($this->stmt,OCI_DEFAULT))
      {
      $this->Print_Error("QueryInOutHash(): Execute query failed!");        
      }
    $this->query_counter++;
    }

  } // EOF
?>
